# app.py
from fastapi import FastAPI, HTTPException, Response, Query
from fastapi.staticfiles import StaticFiles
from fastapi.responses import RedirectResponse
from pydantic import BaseModel, Field
from typing import List, Dict, Any, Optional
import os
from dotenv import load_dotenv
import uvicorn
from datetime import datetime

load_dotenv()

# OpenAI SDK (>=1.x)
from openai import OpenAI
client = OpenAI()  # OPENAI_API_KEY는 .env에서 자동 로드

# Tavily Search API
try:
    from tavily import TavilyClient
    tavily_client = TavilyClient(api_key=os.getenv("TAVILY_API_KEY"))
    TAVILY_ENABLED = True
    print("✅ Tavily 검색 기능 활성화됨")
except Exception as e:
    print(f"⚠️  Tavily 검색 기능 비활성화: {e}")
    tavily_client = None
    TAVILY_ENABLED = False

# Google Custom Search API
GOOGLE_API_KEY = os.getenv("GOOGLE_API_KEY")
GOOGLE_CSE_ID = os.getenv("GOOGLE_CSE_ID")
GOOGLE_SEARCH_ENABLED = bool(GOOGLE_API_KEY and GOOGLE_CSE_ID)

if GOOGLE_SEARCH_ENABLED:
    print("✅ Google Custom Search 활성화됨")
else:
    print("⚠️  Google Custom Search 비활성화 (GOOGLE_API_KEY와 GOOGLE_CSE_ID 필요)")

# Bright Data SERP API
BRIGHTDATA_API_KEY = os.getenv("BRIGHTDATA_API_KEY")
BRIGHTDATA_ZONE = os.getenv("BRIGHTDATA_ZONE", "serp_api1")
BRIGHTDATA_ENABLED = bool(BRIGHTDATA_API_KEY)

if BRIGHTDATA_ENABLED:
    print("✅ Bright Data 검색 기능 활성화됨")
else:
    print("⚠️  Bright Data 검색 기능 비활성화")

# MCP 비즈니스 로직 임포트
from mcp_server import (
    lifespan as mcp_lifespan,
    create_database_analysis_page,
    update_existing_page,
    create_database_table,
    search_notion_pages
)

app = FastAPI(
    title="Notion Automation MCP Web API",
    version="2.0.0",
    lifespan=mcp_lifespan
)

# --- 대화 히스토리 저장 (간단 구현, 실제로는 DB 사용 권장) ---
conversation_histories: Dict[str, List[Dict]] = {}

# --- 정적 파일 서빙 (/static) + 루트 리디렉트 ---
STATIC_DIR = os.path.join(os.path.dirname(__file__), "static")
os.makedirs(STATIC_DIR, exist_ok=True)
app.mount("/static", StaticFiles(directory=STATIC_DIR), name="static")

@app.get("/")
def root():
    return RedirectResponse(url="/static/index.html", status_code=307)

@app.get("/favicon.ico")
def favicon():
    return Response(status_code=204)

# --- 요청 모델 ---
class CreatePageReq(BaseModel):
    title: str = Field(..., description="생성할 페이지 제목")
    analysis_data: str = Field(..., description="분석 결과 텍스트(또는 JSON 문자열)")
    page_id: Optional[str] = Field(
        None,
        description="부모 페이지 ID (없으면 .env의 NOTION_PAGE_ID 사용)"
    )

class UpdatePageReq(BaseModel):
    page_id: str = Field(..., description="업데이트할 페이지 ID")
    analysis_data: str = Field(..., description="추가/덮어쓸 분석 결과")
    append: bool = Field(True, description="True: 추가, False: 기존 블록 archive 후 덮어쓰기")

class CreateTableReq(BaseModel):
    page_id: str = Field(..., description="테이블을 넣을 페이지 ID")
    table_data: List[Dict[str, Any]] = Field(..., description="테이블 데이터(딕셔너리 리스트)")
    table_title: Optional[str] = Field("분석 결과 테이블", description="테이블 제목")

class ChatReq(BaseModel):
    message: str = Field(..., description="사용자 질문")
    session_id: Optional[str] = Field("default", description="대화 세션 ID (대화 히스토리 유지용)")
    page_id: Optional[str] = Field(None, description="저장할 부모 페이지 ID(옵션)")

# --- 키워드 감지 함수 ---
def should_save_to_notion(message: str) -> bool:
    """메시지에 저장 관련 키워드가 있는지 확인"""
    save_keywords = [
        "저장", "노션", "기록", "남겨", "보관",
        "save", "notion", "record", "keep"
    ]
    message_lower = message.lower()
    return any(keyword in message_lower for keyword in save_keywords)

def should_search_web(message: str) -> bool:
    """메시지에 웹 검색이 필요한지 판단"""
    search_keywords = [
        "뉴스", "최신", "검색", "찾아", "알려줘", "today", "news",
        "현재", "지금", "오늘", "실시간", "최근", "날자", "날짜",
        "정치", "경제", "사회", "문화", "스포츠", "국제",
        "주가", "환율", "코인", "비트코인", "주식",
        "날씨", "기온", "미세먼지"
    ]
    message_lower = message.lower()
    has_keyword = any(keyword in message_lower for keyword in search_keywords)
    
    # 디버깅용 로그
    if has_keyword:
        print(f"🔍 검색 키워드 감지: '{message}'")
    
    return has_keyword

def search_web(query: str) -> str:
    """웹 검색 (Tavily > Google > Bright Data 순서)"""
    
    # Tavily 우선 사용 (가장 AI 친화적)
    if TAVILY_ENABLED and tavily_client:
        try:
            print(f"🔎 Tavily로 웹 검색 실행: {query}")
            results = tavily_client.search(query, max_results=5)
            
            search_summary = "\n\n===== 📰 최신 웹 검색 결과 =====\n\n"
            search_summary += "다음은 실시간으로 검색한 최신 정보입니다:\n\n"
            
            for i, result in enumerate(results.get('results', []), 1):
                title = result.get('title', '제목 없음')
                content = result.get('content', '내용 없음')
                url = result.get('url', '')
                
                search_summary += f"[뉴스 {i}]\n"
                search_summary += f"제목: {title}\n"
                search_summary += f"내용: {content}\n"
                search_summary += f"출처: {url}\n\n"
            
            search_summary += "===== 검색 결과 끝 =====\n\n"
            search_summary += "위 검색 결과를 바탕으로 사용자 질문에 답변해주세요."
            
            print(f"✅ Tavily 검색 완료: {len(results.get('results', []))}개 결과")
            return search_summary
        except Exception as e:
            print(f"❌ Tavily 검색 오류: {e}")
    
    # Google Custom Search fallback
    if GOOGLE_SEARCH_ENABLED:
        try:
            print(f"🔎 Google Custom Search로 검색 실행: {query}")
            
            from urllib.parse import quote
            encoded_query = quote(query)
            
            url = f"https://www.googleapis.com/customsearch/v1?key={GOOGLE_API_KEY}&cx={GOOGLE_CSE_ID}&q={encoded_query}&num=5"
            
            response = requests.get(url, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                items = data.get('items', [])
                
                if items:
                    search_summary = "\n\n===== 📰 최신 웹 검색 결과 =====\n\n"
                    for i, item in enumerate(items[:5], 1):
                        search_summary += f"[결과 {i}]\n"
                        search_summary += f"제목: {item.get('title', '제목 없음')}\n"
                        search_summary += f"내용: {item.get('snippet', '내용 없음')}\n"
                        search_summary += f"출처: {item.get('link', '')}\n\n"
                    
                    search_summary += "===== 검색 결과 끝 =====\n"
                    
                    print(f"✅ Google 검색 완료: {len(items)}개 결과")
                    return search_summary
                else:
                    print("⚠️  Google 검색 결과 없음")
            else:
                print(f"❌ Google 검색 오류: {response.status_code}")
                
        except Exception as e:
            print(f"❌ Google 검색 오류: {e}")
    
    # Bright Data fallback (최후의 수단)
    if BRIGHTDATA_ENABLED:
        try:
            print(f"🔎 Bright Data로 웹 검색 실행: {query}")
            
            from urllib.parse import quote
            encoded_query = quote(query)
            
            headers = {
                "Content-Type": "application/json",
                "Authorization": f"Bearer {BRIGHTDATA_API_KEY}"
            }
            
            payload = {
                "zone": BRIGHTDATA_ZONE,
                "url": f"https://www.google.com/search?q={encoded_query}&num=5",
                "format": "raw"
            }
            
            response = requests.post(
                "https://api.brightdata.com/request",
                headers=headers,
                json=payload,
                timeout=30
            )
            
            if response.status_code == 200:
                content = response.text
                import re
                content = re.sub(r'<script[^>]*>.*?</script>', '', content, flags=re.DOTALL)
                content = re.sub(r'<style[^>]*>.*?</style>', '', content, flags=re.DOTALL)
                content = re.sub(r'<[^>]+>', ' ', content)
                content = re.sub(r'\s+', ' ', content)
                content = content[:1500].strip()
                
                if content:
                    print(f"✅ Bright Data 검색 완료")
                    return f"\n\n===== 📰 웹 검색 결과 =====\n\n{content}\n\n===== 검색 결과 끝 =====\n"
                
        except Exception as e:
            print(f"❌ Bright Data 검색 오류: {e}")
    
    print("❌ 사용 가능한 검색 API 없음")
    return None

def extract_title_from_message(message: str, default_title: str = None) -> str:
    """메시지에서 제목 추출 시도 (간단 구현)"""
    import re
    title_pattern = r'(?:제목|타이틀|title)[:：]\s*(.+?)(?:[,，.]|$)'
    match = re.search(title_pattern, message, re.IGNORECASE)
    if match:
        return match.group(1).strip()
    
    return default_title or f"챗 기록 {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}"

# --- 헬스체크 ---
@app.get("/health")
async def health():
    notion_key = os.getenv("NOTION_API_KEY")
    notion_page = os.getenv("NOTION_PAGE_ID")
    
    return {
        "status": "ok",
        "notion_key_loaded": bool(notion_key),
        "notion_key_prefix": notion_key[:10] + "..." if notion_key else None,
        "notion_page_id_loaded": bool(notion_page),
        "notion_page_id": notion_page if notion_page else None,
        "openai_key_loaded": bool(os.getenv("OPENAI_API_KEY")),
        "google_search_enabled": GOOGLE_SEARCH_ENABLED,
        "brightdata_search_enabled": BRIGHTDATA_ENABLED,
        "tavily_search_enabled": TAVILY_ENABLED
    }

# --- 개선된 OpenAI + 자동 Notion 저장 + 웹 검색 ---
@app.post("/chat")
async def chat(req: ChatReq):
    """
    자연스러운 대화를 제공하며, 
    - 검색 키워드가 있으면 자동으로 웹 검색
    - 저장 키워드가 있으면 자동으로 Notion에 저장
    """
    session_id = req.session_id or "default"
    
    # 1) 대화 히스토리 가져오기 (없으면 초기화)
    if session_id not in conversation_histories:
        conversation_histories[session_id] = []
    
    history = conversation_histories[session_id]
    
    # 2) 저장 여부 자동 감지
    should_save = should_save_to_notion(req.message)
    
    # 2.5) 웹 검색이 필요한지 판단
    needs_search = should_search_web(req.message)
    search_context = ""
    
    if needs_search and (GOOGLE_SEARCH_ENABLED or BRIGHTDATA_ENABLED or TAVILY_ENABLED):
        print(f"🔍 검색 실행 중...")
        search_results = search_web(req.message)
        if search_results:
            search_context = search_results
            print(f"✅ 검색 결과 획득")
        else:
            print(f"⚠️ 검색 결과 없음")
    elif needs_search:
        print(f"⚠️ 검색 필요하지만 검색 API 비활성화됨")
    
    # 3) 시스템 메시지 구성 (현재 날짜 정보 포함)
    current_date = datetime.now().strftime("%Y년 %m월 %d일 %A")
    # 요일을 한국어로 변환
    weekday_kr = {
        "Monday": "월요일", "Tuesday": "화요일", "Wednesday": "수요일",
        "Thursday": "목요일", "Friday": "금요일", "Saturday": "토요일", "Sunday": "일요일"
    }
    current_date_kr = current_date
    for en, kr in weekday_kr.items():
        current_date_kr = current_date_kr.replace(en, kr)
    
    system_message = {
        "role": "system",
        "content": (
            f"당신은 친근하고 도움이 되는 한국어 어시스턴트입니다. "
            f"사용자와 자연스럽게 대화하세요.\n"
            f"오늘 날짜는 {current_date_kr}입니다.\n"
            f"{chr(10) if search_context else ''}"
            f"{'🔍 중요: 사용자 메시지에 웹 검색 결과가 포함되어 있습니다. 반드시 이 최신 검색 결과를 바탕으로 구체적이고 정확하게 답변하세요. 검색 결과에 있는 제목, 내용, 출처를 활용하여 최신 정보를 제공해주세요. 절대 \"검색 결과가 없다\" 또는 \"확인할 수 없다\"고 하지 마세요.' if search_context else ''}"
        )
    }
    
    # 4) 대화 히스토리에 현재 메시지 추가 (검색 컨텍스트 포함)
    user_message_content = req.message
    if search_context:
        # 검색 결과를 명확하게 추가
        user_message_content = f"{req.message}\n\n{search_context}"
        print(f"📝 검색 결과를 메시지에 추가 (총 {len(search_context)} 문자)")
    
    history.append({"role": "user", "content": user_message_content})
    
    # 5) OpenAI 호출 (히스토리 포함, 최근 10개만)
    try:
        messages = [system_message] + history[-10:]  # 최근 10개 대화만 유지
        completion = client.chat.completions.create(
            model="gpt-4o-mini",
            messages=messages,
            temperature=0.7,
        )
        answer = completion.choices[0].message.content or ""
        
        # 6) 히스토리에 답변 추가
        history.append({"role": "assistant", "content": answer})
        
    except Exception as e:
        error_msg = str(e)
        
        # 429 에러 (Rate Limit) 처리
        if "429" in error_msg or "rate_limit" in error_msg.lower():
            answer = (
                "😓 죄송합니다. OpenAI API 요청 한도를 초과했습니다.\n\n"
                "**원인:**\n"
                "• 무료 크레딧이 소진되었거나\n"
                "• 분당 요청 제한을 초과했습니다\n\n"
                "**해결 방법:**\n"
                "1. https://platform.openai.com/usage 에서 크레딧 확인\n"
                "2. 결제 정보를 등록하면 한도가 증가합니다\n"
                "3. 잠시 후 다시 시도해주세요"
            )
        # 인증 에러
        elif "401" in error_msg or "authentication" in error_msg.lower():
            answer = (
                "🔑 API 키 인증 오류가 발생했습니다.\n\n"
                ".env 파일의 OPENAI_API_KEY를 확인해주세요.\n"
                "https://platform.openai.com/api-keys 에서 새 키를 발급받을 수 있습니다."
            )
        else:
            answer = f"❌ OpenAI API 오류가 발생했습니다:\n{error_msg}"
        
        history.append({"role": "assistant", "content": answer})

    result = {
        "success": True,
        "answer": answer,
        "saved_to_notion": False,
        "session_id": session_id,
        "search_executed": bool(search_context)
    }

    # 7) 저장 키워드가 있으면 자동으로 Notion에 저장
    if should_save:
        try:
            # 제목 추출 시도
            title = extract_title_from_message(req.message)
            
            # 대화 내용 포맷팅 (최근 대화 포함)
            conversation_text = ""
            recent_history = history[-6:]  # 최근 3턴 (Q&A 3쌍)
            
            for i, msg in enumerate(recent_history):
                if msg["role"] == "user":
                    conversation_text += f"Q: {msg['content']}\n\n"
                elif msg["role"] == "assistant":
                    conversation_text += f"A: {msg['content']}\n\n"
            
            # Notion에 저장
            notion_res = await create_database_analysis_page(
                title, 
                conversation_text.strip(), 
                req.page_id
            )
            
            result["notion"] = notion_res
            
            # 저장 완료 메시지 추가 (성공했을 때만)
            if notion_res.get("success"):
                result["saved_to_notion"] = True
                save_confirmation = f"\n\n✅ 대화 내용이 Notion에 저장되었습니다!\n페이지: {title}"
                result["answer"] += save_confirmation
            else:
                # 저장 실패 시 에러 메시지 추가
                error_detail = notion_res.get("error", "알 수 없는 오류")
                error_message = f"\n\n❌ Notion 저장 실패: {error_detail}"
                result["answer"] += error_message
                
        except Exception as e:
            result["notion"] = {"success": False, "error": str(e)}
            error_message = f"\n\n❌ Notion 저장 중 오류 발생: {str(e)}"
            result["answer"] += error_message

    return result

# --- 대화 히스토리 초기화 ---
@app.post("/chat/reset")
async def reset_conversation(session_id: str = "default"):
    """특정 세션의 대화 히스토리 초기화"""
    if session_id in conversation_histories:
        conversation_histories[session_id] = []
    return {
        "success": True,
        "message": f"세션 '{session_id}'의 대화 히스토리가 초기화되었습니다.",
        "session_id": session_id
    }

# --- Notion 도구 API (기존 유지) ---
@app.post("/notion/create")
async def http_create_page(req: CreatePageReq):
    try:
        result = await create_database_analysis_page(req.title, req.analysis_data, req.page_id)
        if not result.get("success", False):
            raise HTTPException(status_code=500, detail=result.get("error", "생성 실패"))
        return result
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/notion/update")
async def http_update_page(req: UpdatePageReq):
    try:
        result = await update_existing_page(req.page_id, req.analysis_data, req.append)
        if not result.get("success", False):
            raise HTTPException(status_code=500, detail=result.get("error", "업데이트 실패"))
        return result
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/notion/table")
async def http_create_table(req: CreateTableReq):
    try:
        result = await create_database_table(req.page_id, req.table_data, req.table_title)
        if not result.get("success", False):
            raise HTTPException(status_code=500, detail=result.get("error", "테이블 생성 실패"))
        return result
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@app.get("/notion/search")
async def http_search(query: str = "", page_size: int = 10):
    try:
        result = await search_notion_pages(query, page_size)
        if not result.get("success", False):
            raise HTTPException(status_code=500, detail=result.get("error", "검색 실패"))
        return result
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

# --- Notion 연결 테스트 ---
@app.get("/test-notion")
async def test_notion():
    """Notion API 연결 및 설정 테스트"""
    notion_key = os.getenv("NOTION_API_KEY")
    notion_page = os.getenv("NOTION_PAGE_ID")
    
    result = {
        "checks": [],
        "success": True
    }
    
    # 1. API 키 확인
    if not notion_key:
        result["checks"].append({
            "test": "NOTION_API_KEY",
            "status": "❌ FAILED",
            "message": ".env 파일에 NOTION_API_KEY가 없습니다"
        })
        result["success"] = False
    else:
        result["checks"].append({
            "test": "NOTION_API_KEY",
            "status": "✅ PASSED",
            "message": f"키 발견: {notion_key[:15]}..."
        })
    
    # 2. Page ID 확인
    if not notion_page:
        result["checks"].append({
            "test": "NOTION_PAGE_ID",
            "status": "⚠️ WARNING",
            "message": ".env 파일에 NOTION_PAGE_ID가 없습니다 (기본 페이지 사용 불가)"
        })
    else:
        result["checks"].append({
            "test": "NOTION_PAGE_ID",
            "status": "✅ PASSED",
            "message": f"페이지 ID: {notion_page}"
        })
    
    # 3. 실제 Notion API 호출 테스트
    if notion_key and notion_page:
        try:
            test_result = await create_database_analysis_page(
                "🧪 연결 테스트",
                "이 페이지는 Notion API 연결 테스트용입니다.\n" + 
                f"테스트 시간: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}",
                notion_page
            )
            
            if test_result.get("success"):
                result["checks"].append({
                    "test": "Notion API 호출",
                    "status": "✅ PASSED",
                    "message": "테스트 페이지가 성공적으로 생성되었습니다!",
                    "url": test_result.get("url")
                })
            else:
                result["checks"].append({
                    "test": "Notion API 호출",
                    "status": "❌ FAILED",
                    "message": test_result.get("error", "알 수 없는 오류")
                })
                result["success"] = False
                
        except Exception as e:
            result["checks"].append({
                "test": "Notion API 호출",
                "status": "❌ FAILED",
                "message": str(e)
            })
            result["success"] = False
    
    return result

# --- 검색 기능 테스트 ---
@app.get("/test-search")
async def test_search(query: str = "latest AI news"):
    """웹 검색 기능 테스트"""
    if not GOOGLE_SEARCH_ENABLED and not BRIGHTDATA_ENABLED and not TAVILY_ENABLED:
        return {
            "success": False,
            "error": "검색 API가 비활성화되어 있습니다."
        }
    
    try:
        results = search_web(query)
        
        used_api = "None"
        if GOOGLE_SEARCH_ENABLED:
            used_api = "Google Custom Search"
        elif BRIGHTDATA_ENABLED:
            used_api = "Bright Data"
        elif TAVILY_ENABLED:
            used_api = "Tavily"
        
        return {
            "success": bool(results),
            "query": query,
            "results": results,
            "message": "검색 성공!" if results else "검색 실패",
            "available_apis": {
                "google": GOOGLE_SEARCH_ENABLED,
                "brightdata": BRIGHTDATA_ENABLED,
                "tavily": TAVILY_ENABLED
            },
            "used_api": used_api
        }
    except Exception as e:
        return {
            "success": False,
            "error": str(e)
        }

if __name__ == "__main__":
    # 기본 포트 8000
    uvicorn.run(app, host="0.0.0.0", port=8000)
