# mcp_server.py
from mcp.server.fastmcp import FastMCP
from mcp.server.fastmcp.prompts import base
from contextlib import asynccontextmanager
import os
from dotenv import load_dotenv
from datetime import datetime
from typing import Dict, List, Any, Optional
import requests

# 환경 변수 로드
load_dotenv()

# 전역 Notion API 설정
NOTION_API_KEY = None
NOTION_VERSION = "2022-06-28"

@asynccontextmanager
async def lifespan(app):
    """서버 시작/종료 시 Notion API 설정 관리"""
    global NOTION_API_KEY
    try:
        NOTION_API_KEY = os.getenv("NOTION_API_KEY")
        if not NOTION_API_KEY:
            raise ValueError("NOTION_API_KEY 환경 변수가 설정되지 않았습니다.")
        print("Notion API 설정 완료")
        yield
    finally:
        NOTION_API_KEY = None
        print("Notion API 연결 종료")

def _make_notion_request(method: str, endpoint: str, data: Dict = None) -> Dict:
    """Notion API 요청을 보내는 헬퍼 함수 (에러 바디 포함)"""
    if NOTION_API_KEY is None:
        raise ValueError("Notion API 키가 설정되지 않았습니다.")
    
    headers = {
        "Authorization": f"Bearer {NOTION_API_KEY}",
        "Content-Type": "application/json",
        "Notion-Version": NOTION_VERSION
    }
    url = f"https://api.notion.com/v1/{endpoint}"

    response = None
    try:
        method = method.upper()
        if method == "GET":
            response = requests.get(url, headers=headers)
        elif method == "POST":
            response = requests.post(url, headers=headers, json=data)
        elif method == "PATCH":
            response = requests.patch(url, headers=headers, json=data)
        elif method == "DELETE":
            # 일반적으로 Notion에서는 블록 삭제 대신 archive(PATCH)를 권장
            response = requests.delete(url, headers=headers)
        else:
            raise ValueError(f"지원하지 않는 HTTP 메서드: {method}")
        
        response.raise_for_status()
        if response.status_code == 204:
            return {}
        return response.json()
    except requests.exceptions.HTTPError as e:
        body = ""
        try:
            body = response.text if response is not None else ""
        except Exception:
            pass
        raise Exception(f"Notion API 요청 실패: {e}; body={body}")
    except requests.exceptions.RequestException as e:
        raise Exception(f"Notion API 네트워크 오류: {str(e)}")

mcp = FastMCP("NotionAutomation", lifespan=lifespan)

@mcp.tool()
async def create_database_analysis_page(
    title: str,
    analysis_data: str,
    page_id: Optional[str] = None
) -> Dict[str, Any]:
    """
    데이터베이스 분석 결과를 Notion 페이지에 자동으로 생성합니다.
    parent는 반드시 page_id여야 하며(루트 생성 금지),
    인자로 없으면 .env의 NOTION_PAGE_ID를 사용합니다.
    """
    if NOTION_API_KEY is None:
        raise ValueError("Notion API 키가 설정되지 않았습니다.")
    
    try:
        # 부모 페이지 강제 (.env 또는 인자)
        parent_id = page_id or os.getenv("NOTION_PAGE_ID")
        if not parent_id:
            raise ValueError("parent page_id가 필요합니다 (.env NOTION_PAGE_ID 또는 인자로 전달).")

        blocks = _convert_analysis_to_blocks(analysis_data)

        page_data = {
            "parent": {"page_id": parent_id},   # 루트 금지
            "properties": {
                "title": {
                    "title": [{"text": {"content": title}}]
                }
            },
            "children": blocks
        }
        result = _make_notion_request("POST", "pages", page_data)

        return {
            "success": True,
            "page_id": result.get("id"),
            "url": result.get("url"),
            "title": title,
            "message": "데이터베이스 분석 결과가 Notion 페이지에 성공적으로 생성되었습니다."
        }
    except Exception as e:
        return {
            "success": False,
            "error": f"페이지 생성 중 오류 발생: {str(e)}"
        }

@mcp.tool()
async def update_existing_page(
    page_id: str,
    analysis_data: str,
    append: bool = True
) -> Dict[str, Any]:
    """
    기존 Notion 페이지에 데이터베이스 분석 결과를 추가/덮어쓰기 합니다.
    append=False일 경우 기존 블록을 archive 후 새 블록을 추가합니다.
    """
    if NOTION_API_KEY is None:
        raise ValueError("Notion API 키가 설정되지 않았습니다.")
    
    try:
        blocks = _convert_analysis_to_blocks(analysis_data)

        if not append:
            # 기존 블록 조회
            existing = _make_notion_request("GET", f"blocks/{page_id}/children")
            for block in existing.get("results", []):
                # 블록 archive (삭제 대신)
                _make_notion_request("PATCH", f"blocks/{block['id']}", {"archived": True})

        # 새 블록들 추가 (대량 추가)
        _make_notion_request("PATCH", f"blocks/{page_id}/children", {"children": blocks})

        return {
            "success": True,
            "page_id": page_id,
            "message": "페이지가 성공적으로 업데이트되었습니다."
        }
    except Exception as e:
        return {
            "success": False,
            "error": f"페이지 업데이트 중 오류 발생: {str(e)}"
        }

@mcp.tool()
async def create_database_table(
    page_id: str,
    table_data: List[Dict[str, Any]],
    table_title: str = "분석 결과 테이블"
) -> Dict[str, Any]:
    """
    데이터베이스 쿼리 결과를 Notion 페이지에 테이블로 생성합니다.
    """
    if NOTION_API_KEY is None:
        raise ValueError("Notion API 키가 설정되지 않았습니다.")
    
    try:
        if not table_data:
            return {"success": False, "error": "테이블 데이터가 비어있습니다."}

        headers = list(table_data[0].keys())

        title_block = {
            "type": "heading_2",
            "heading_2": {
                "rich_text": [{"type": "text", "text": {"content": table_title}}]
            }
        }

        table_block = {
            "type": "table",
            "table": {
                "table_width": len(headers),
                "has_column_header": True,
                "has_row_header": False,
                "children": []
            }
        }

        # 헤더 행
        header_row = {
            "type": "table_row",
            "table_row": {
                "cells": [[{"type": "text", "text": {"content": str(h)}}] for h in headers]
            }
        }
        table_block["table"]["children"].append(header_row)

        # 데이터 행
        for row in table_data:
            table_block["table"]["children"].append({
                "type": "table_row",
                "table_row": {
                    "cells": [[{"type": "text", "text": {"content": "" if v is None else str(v)}}]
                              for v in row.values()]
                }
            })

        # 한 번에 추가
        _make_notion_request("PATCH", f"blocks/{page_id}/children", {"children": [title_block, table_block]})

        return {
            "success": True,
            "page_id": page_id,
            "table_title": table_title,
            "rows": len(table_data),
            "columns": len(headers),
            "message": "테이블이 성공적으로 생성되었습니다."
        }
    except Exception as e:
        return {
            "success": False,
            "error": f"테이블 생성 중 오류 발생: {str(e)}"
        }

@mcp.tool()
async def search_notion_pages(query: str = "", page_size: int = 10) -> Dict[str, Any]:
    """
    Notion에서 페이지를 검색합니다.
    """
    if NOTION_API_KEY is None:
        raise ValueError("Notion API 키가 설정되지 않았습니다.")
    
    try:
        search_data = {"query": query, "page_size": page_size}
        result = _make_notion_request("POST", "search", search_data)

        pages = []
        for page in result.get("results", []):
            if page.get("object") == "page":
                pages.append({
                    "id": page["id"],
                    "title": _extract_page_title(page),
                    "url": page["url"],
                    "created_time": page.get("created_time"),
                    "last_edited_time": page.get("last_edited_time")
                })

        return {"success": True, "pages": pages, "total": len(pages), "query": query}
    except Exception as e:
        return {"success": False, "error": f"페이지 검색 중 오류 발생: {str(e)}"}

def _convert_analysis_to_blocks(analysis_data: str) -> List[Dict[str, Any]]:
    """긴 텍스트를 안전한 크기로 분할하여 Paragraph 블록으로 변환"""
    blocks = []
    current_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

    # 헤더
    blocks.append({
        "type": "heading_1",
        "heading_1": {
            "rich_text": [{"type": "text", "text": {"content": f"데이터베이스 분석 결과 - {current_time}"}}]
        }
    })

    # 텍스트 분할
    chunk_size = 1900
    text_chunks = [analysis_data[i:i+chunk_size] for i in range(0, len(analysis_data), chunk_size)] or [""]
    for chunk in text_chunks:
        blocks.append({
            "type": "paragraph",
            "paragraph": {
                "rich_text": [{"type": "text", "text": {"content": chunk}}]
            }
        })
    return blocks

def _extract_page_title(page: Dict[str, Any]) -> str:
    """페이지에서 제목 추출"""
    properties = page.get("properties", {})
    for _, prop_data in properties.items():
        if prop_data.get("type") == "title" and prop_data.get("title"):
            return prop_data["title"][0]["text"]["content"]
    return "제목 없음"

@mcp.resource("notion://pages")
async def get_notion_pages() -> Dict[str, Any]:
    """Notion 페이지 목록 조회 (간단 조회)"""
    if NOTION_API_KEY is None:
        return {"error": "Notion API 키가 설정되지 않았습니다."}
    try:
        result = _make_notion_request("POST", "search", {"page_size": 20})
        pages = []
        for page in result.get("results", []):
            if page.get("object") == "page":
                pages.append({
                    "id": page["id"],
                    "title": _extract_page_title(page),
                    "url": page["url"]
                })
        return {"pages": pages, "total": len(pages)}
    except Exception as e:
        return {"error": f"페이지 조회 중 오류: {str(e)}"}

@mcp.prompt()
def default_prompt(message: str) -> List[base.Message]:
    return [
        base.AssistantMessage(
            "당신은 Notion 자동화 어시스턴트입니다.\n"
            "- 데이터베이스 분석 결과를 Notion 페이지에 자동 생성\n"
            "- 기존 페이지에 분석 결과 추가/업데이트\n"
            "- 쿼리 결과를 Notion 테이블로 변환\n"
            "- Notion 페이지 검색 및 관리\n"
            "분석 결과를 구조화된 형태로 Notion에 저장합니다."
        ),
        base.UserMessage(message),
    ]

if __name__ == "__main__":
    try:
        print("Notion Automation MCP Server is running...")
        mcp.run(transport="stdio")
    except KeyboardInterrupt:
        print("\n✅ 서버가 정상적으로 종료되었습니다.")
    except Exception as e:
        print(f"❌ 서버 오류: {e}")

